function [rmse, cdx_new] = wrapper_tranche_mispricing(x0, CDX, discounts_IMM, start_date_num, end_date_num, N, ...
                                                      maturities_cds, maturities_tranches, RMSE_method, LGD_method, how_port_loss, method5, method2, CDX2)
% --------------------------------------------------------------------------------------------------
% For a set of parameters of the common factor, fit the individual CDS prices, calculate the model
% implied tranche prices (between start_date and end_date) and return the RMSE pricing error.
% --------------------------------------------------------------------------------------------------
% x0                            ... [k, theta_total_avg, sigma, L_total, mu, omega1], i.e. new parameters for AJD dynamics of common factor
% CDX                           ... credit index structure (see 'all_steps_in_a_row.m')
% discounts_IMM                 ... structure with discount curves matching IMM dates
% start_date_num                ... datenum of start date
% end_date_num                  ... datenum of end date
% N                             ... number of points for numerical integration of Fourier transform
% maturities_cds                ... which maturities to fit for CDS prices, default: all
% maturities_tranches           ... which maturities to fit for tranche prices, default: all
% RMSE_method                   ... which definition of pricing error to use
% LGD_method                    ... method for (joint) distribution of LGDs
% how_port_loss                 ... how to calculate the portfolio loss distribution
%                                   1 ... load from external file
%                                   2 ... calculate
%                                   3 ... calculate and save in external file
% method5                       ... which definition to use for the 5th parameters
%                                   'mu' = expected jump size \mu (default)
%                                   'mu*L' = jump intensity x jump size
% method2                       ... which definition to use for the 2nd parameter
%                               ... 'thet' = theta_total
%                               ... 'thet*k' = theta_total * k
% CDX2                          ... second CDX structure for robust RMSE calculation (optional)
% --------------------------------------------------------------------------------------------------
% sample call:
%{
wrapper_tranche_mispricing({0.25 0.02 0.001 0 0.03 0.8 0.9 CDX_NA_IG2_monthly.dates{1}}, CDX_NA_IG2_monthly, discounts_IMM, datenum('02/01/2006'), ...
                           datenum('03/01/2006'), 2000, [1 0 0 0], [1 0 0], 1, 0, 2)
%}
% --------------------------------------------------------------------------------------------------

% Determine date-range that is updated
%CDX = normalize_ai(CDX);
start_pos = find(CDX.dates{1} >= start_date_num, 1, 'first');
end_pos = find(CDX.dates{1} <= end_date_num, 1, 'last');
used_dates = CDX.dates{1}(start_pos:end_pos);

% Extract new parameters for AJD of common factor
% disp(['x0{1}: ' num2str(x0{1}')]);
% disp(['x0{2}: ' num2str(x0{2}')]);
% disp(['x0{3}: ' num2str(x0{3}')]);
% disp(['x0{4}: ' num2str(x0{4}')]);
% disp(['x0{5}: ' num2str(x0{5}')]);
% disp(['x0{6}: ' num2str(x0{6}')]);
% disp(['x0{7}: ' num2str(x0{7}')]);
k = x0{1};
if strcmp(method2, 'thet')
    theta_total_avg = abs(x0{2}) .* sign(k);
elseif strcmp(method2, 'thet*k')
    theta_total_avg = abs(x0{2}) ./ k;
end
sigma = abs(x0{3});
L_total = abs(x0{4});
if strcmp(method5, 'mu')
    mu = abs(x0{5});
elseif strcmp(method5, 'mu*L')
    mu = abs(x0{5}) ./ L_total;
end
omega1 = abs(x0{6});
omega2 = abs(x0{7});

% Set new parameters
x0_old = get_x0(CDX, used_dates, method5, method2);
CDX = set_k(CDX, k, used_dates);
CDX = set_sigma(CDX, sigma, used_dates);
CDX = set_L_total(CDX, L_total, used_dates); 
CDX = set_mu(CDX, mu, used_dates);
CDX = set_omega1(CDX, omega1, used_dates); 
CDX = set_theta_omega2(CDX, theta_total_avg, omega2, used_dates);
%CDX = set_y0(CDX, omega2);

% Update fit of individual CDS (but only if portfolio loss distribution is calculated internally)
if (how_port_loss >= 2)
    % Do not update CDS fit if only omega1 has changed
    % Determine parameters that have changed
    new_entry = zeros(7,1);
    for i=1:7
        new_entry(i) = sum(abs(x0_old{i} - x0{i})) > 1e-10;
    end
    tmp = find(logical(new_entry));
    if (sum(is_member_sorted_c([1 2 3 4 5 7], tmp)) > 0) | (sum(is_member_sorted_c([6 8], tmp)) ~= 1)
        %a_old = get_portfolio_parameters(CDX, CDX.dates{1}(1), CDX.dates{1}(end));
        %[x_it_old, lambda_it_old] = get_x_it(CDX, CDX.dates{1}(2));
        CDX = fit_model_cds_portfolio(CDX, discounts_IMM, start_date_num, end_date_num, maturities_cds);
        %a_new = get_portfolio_parameters(CDX, CDX.dates{1}(1), CDX.dates{1}(end));
        %[x_it_new, lambda_it_new] = get_x_it(CDX, CDX.dates{1}(2));
        % abs(a_old(2,:) - a_new(2,:))
        % abs(x_it_old - x_it_new)
        % abs(lambda_it_old - lambda_it_new)
    end
end

% Update model-implied index price
CDX = update_model_price_index(CDX, discounts_IMM, maturities_tranches, start_date_num, end_date_num);

% Update model-implied tranche prices
if (1) | (length(used_dates) == 1)      % Single processor version
    CDX = update_model_price_tranches(CDX, discounts_IMM, N, start_date_num, end_date_num, maturities_tranches, LGD_method, how_port_loss);
else                                    % Distribute computation across multiple processors
    % Create and submit sub-tasks
    cut_off = floor(length(used_dates)/2);
    jm = findResource('scheduler','type','jobmanager');
    j = createJob(jm);
    createTask(j, @update_model_price_tranches, 1, {CDX, discounts_IMM, N, start_date_num, used_dates(cut_off), maturities_tranches, ...
                                                    LGD_method, how_port_loss});
    createTask(j, @update_model_price_tranches, 1, {CDX, discounts_IMM, N, used_dates(cut_off+1), end_date_num, maturities_tranches, ...
                                                    LGD_method, how_port_loss});
    
    % Submit tasks and wait for results
    submit(j);
    waitForState(j);
    results = getAllOutputArguments(j);
    destroy(j);
    
    % Combine results from sub-tasks
    if isempty(results{1}) | isempty(results{2})
        error('Distributed computing job returned empty results.');
    end
    CDX = CDX_copy_parameters(CDX, results{1}, used_dates(1:cut_off));
    CDX = CDX_copy_parameters(CDX, results{2}, used_dates((cut_off+1):end));
end 

% Split RMSE method into part for RMSE_INDEX (third digit after comma), RMSE_CDS (second digit after comma) and RMSE_TR (first digit)
RMSE_method_index = floor(RMSE_method / 100);
RMSE_method = RMSE_method - RMSE_method_index*100;
RMSE_method_cds = floor(RMSE_method / 10);
RMSE_method_tr = RMSE_method - RMSE_method_cds*10;

% Add RMSE_TRANCHES
if (RMSE_method_tr <= 3)
    [rms_errors, cdx_new] = RMSE_tranche_mispricing2(CDX, start_date_num, end_date_num, maturities_tranches, RMSE_method_tr, (how_port_loss~=1));
elseif (RMSE_method_tr == 4)
    [rms_errors, cdx_new] = RMSE_tranche_mispricing2(CDX, start_date_num, end_date_num, maturities_tranches, RMSE_method_tr, (how_port_loss~=1), CDX2);
end

% Add RMSE_CDS
if (RMSE_method_cds == 1)
    rms_errors = rms_errors + cdx_new.rmse_avg_cds(start_pos:end_pos);
elseif (RMSE_method_cds == 2)
    rms_errors = rms_errors + cdx_new.rmse_5yr_cds(start_pos:end_pos);
end

% Add RMSE_INDEX
if (RMSE_method_index == 1)
    rms_errors = rms_errors + 5 * cdx_new.rmse_index(start_pos:end_pos);    
end

% Insert total RMSE
cdx_new.rmse(start_pos:end_pos) = rms_errors;
if (how_port_loss ~= 1) & (sum(new_entry) == 0)
    rmse = -1;
else
    rmse = sqrt(sum(rms_errors.^2) / length(rms_errors));
end
avg_rms_error = mean(rms_errors);
disp(['RMSE_total: ' num2str(avg_rms_error)]);

